import { useState, useEffect, useRef, useCallback } from "react";

const GLOBAL_CSS = `
  @import url('https://fonts.googleapis.com/css2?family=Sora:wght@300;400;500;600;700&family=JetBrains+Mono:wght@400;500;600&display=swap');

  *, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }

  html, body, #root {
    height: 100%;
    width: 100%;
  }

  body {
    background: #0F1117;
    font-family: 'Sora', sans-serif;
    color: #E8E6F0;
    -webkit-font-smoothing: antialiased;
  }

  .me-root {
    min-height: 100vh;
    width: 100%;
    background: #0F1117;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 24px 16px;
  }

  .me-card {
    width: 100%;
    max-width: 420px;
    background: #161923;
    border: 1px solid rgba(255,255,255,0.07);
    border-radius: 24px;
    padding: 32px 24px;
  }

  @keyframes fadeUp {
    from { opacity:0; transform:translateY(16px); }
    to   { opacity:1; transform:translateY(0);    }
  }
  @keyframes matchPop {
    0%   { transform: scale(1); }
    40%  { transform: scale(1.12); }
    100% { transform: scale(1); }
  }
  @keyframes wrongShake {
    0%,100% { transform:translateX(0); }
    20% { transform:translateX(-6px); }
    40% { transform:translateX(6px); }
    60% { transform:translateX(-4px); }
    80% { transform:translateX(4px); }
  }
  @keyframes timerBeat {
    0%,100% { transform:scale(1); }
    50%      { transform:scale(1.1); }
  }
  @keyframes pulse-ring {
    0%   { box-shadow: 0 0 0 0 rgba(52,211,153,0.4); }
    70%  { box-shadow: 0 0 0 8px rgba(52,211,153,0); }
    100% { box-shadow: 0 0 0 0 rgba(52,211,153,0); }
  }

  .screen-anim { animation: fadeUp 0.4s cubic-bezier(0.16,1,0.3,1) both; }

  ::-webkit-scrollbar { width:4px; }
  ::-webkit-scrollbar-thumb { background:#2A2D3A; border-radius:4px; }
`;

const EMOJIS = [
  "🌸",
  "🎯",
  "⚡",
  "🌊",
  "🔥",
  "🍀",
  "🎸",
  "🦋",
  "🌙",
  "🎲",
  "🦊",
  "🌺",
  "🏆",
  "🎭",
  "🦄",
  "🍉",
];

const LEVELS = [
  { id: "easy", label: "Easy", pairs: 6, cols: 4, time: 90 },
  { id: "medium", label: "Medium", pairs: 8, cols: 4, time: 75 },
  { id: "hard", label: "Hard", pairs: 10, cols: 5, time: 60 },
  { id: "expert", label: "Expert", pairs: 12, cols: 6, time: 50 },
];

function buildDeck(pairs) {
  const emojis = [...EMOJIS].sort(() => Math.random() - 0.5).slice(0, pairs);
  const cards = [...emojis, ...emojis].map((e, i) => ({
    id: i,
    emoji: e,
    flipped: false,
    matched: false,
  }));
  for (let i = cards.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [cards[i], cards[j]] = [cards[j], cards[i]];
  }
  return cards;
}

function fmt(s) {
  return `${String(Math.floor(s / 60)).padStart(2, "0")}:${String(s % 60).padStart(2, "0")}`;
}

// ── Card ─────────────────────────────────────────────────────────────────────
function Card({ card, onClick, disabled, size, isWrong }) {
  const visible = card.flipped || card.matched;
  return (
    <div
      onClick={() =>
        !disabled && !card.flipped && !card.matched && onClick(card.id)
      }
      style={{
        width: size,
        height: size,
        cursor:
          disabled || card.flipped || card.matched ? "default" : "pointer",
        perspective: 800,
        flexShrink: 0,
      }}
    >
      <div
        style={{
          width: "100%",
          height: "100%",
          position: "relative",
          transformStyle: "preserve-3d",
          transform: visible ? "rotateY(180deg)" : "rotateY(0deg)",
          transition: "transform 0.4s cubic-bezier(0.4,0,0.2,1)",
          borderRadius: 12,
          animation: card.matched ? "matchPop 0.4s ease" : "none",
        }}
      >
        {/* Back */}
        <div
          style={{
            position: "absolute",
            inset: 0,
            borderRadius: 12,
            backfaceVisibility: "hidden",
            WebkitBackfaceVisibility: "hidden",
            background: isWrong
              ? "rgba(239,68,68,0.2)"
              : "rgba(255,255,255,0.04)",
            border: `1.5px solid ${isWrong ? "rgba(239,68,68,0.5)" : "rgba(255,255,255,0.08)"}`,
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            transition: "background 0.2s, border 0.2s",
          }}
        >
          <div
            style={{
              width: 28,
              height: 28,
              borderRadius: 6,
              background: "rgba(139,92,246,0.15)",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
            }}
          >
            <div
              style={{
                width: 12,
                height: 12,
                borderRadius: 3,
                background: "rgba(139,92,246,0.3)",
              }}
            />
          </div>
        </div>
        {/* Front */}
        <div
          style={{
            position: "absolute",
            inset: 0,
            borderRadius: 12,
            backfaceVisibility: "hidden",
            WebkitBackfaceVisibility: "hidden",
            transform: "rotateY(180deg)",
            background: card.matched
              ? "linear-gradient(145deg,#0c2a1e,#102a20)"
              : "linear-gradient(145deg,#1a1030,#221540)",
            border: `1.5px solid ${card.matched ? "rgba(52,211,153,0.4)" : "rgba(139,92,246,0.35)"}`,
            display: "flex",
            alignItems: "center",
            justifyContent: "center",
            animation: card.matched ? "pulse-ring 0.6s ease" : "none",
          }}
        >
          <span
            style={{
              fontSize: size * 0.42,
              lineHeight: 1,
              filter: card.matched
                ? "drop-shadow(0 0 6px rgba(52,211,153,0.7))"
                : "drop-shadow(0 2px 6px rgba(139,92,246,0.6))",
            }}
          >
            {card.emoji}
          </span>
        </div>
      </div>
    </div>
  );
}

// ── Timer Arc ─────────────────────────────────────────────────────────────────
function TimerArc({ left, total }) {
  const pct = left / total;
  const r = 22,
    c = 2 * Math.PI * r;
  const color = pct > 0.5 ? "#8B5CF6" : pct > 0.25 ? "#F59E0B" : "#EF4444";
  return (
    <div style={{ position: "relative", width: 56, height: 56 }}>
      <svg
        width={56}
        height={56}
        style={{ transform: "rotate(-90deg)", position: "absolute", inset: 0 }}
      >
        <circle
          cx={28}
          cy={28}
          r={r}
          fill="none"
          stroke="#1E2130"
          strokeWidth={4}
        />
        <circle
          cx={28}
          cy={28}
          r={r}
          fill="none"
          stroke={color}
          strokeWidth={4}
          strokeDasharray={c}
          strokeDashoffset={c * (1 - pct)}
          strokeLinecap="round"
          style={{ transition: "stroke-dashoffset 1s linear, stroke 0.5s" }}
        />
      </svg>
      <div
        style={{
          position: "absolute",
          inset: 0,
          display: "flex",
          alignItems: "center",
          justifyContent: "center",
        }}
      >
        <span
          style={{
            fontFamily: "'JetBrains Mono',monospace",
            fontSize: 12,
            fontWeight: 600,
            color,
            animation: pct < 0.25 ? "timerBeat 0.7s ease infinite" : "none",
          }}
        >
          {left}
        </span>
      </div>
    </div>
  );
}

// ── Stat Box ──────────────────────────────────────────────────────────────────
function StatBox({ label, value, color = "#E8E6F0", sub }) {
  return (
    <div
      style={{
        flex: 1,
        background: "rgba(255,255,255,0.03)",
        border: "1px solid rgba(255,255,255,0.07)",
        borderRadius: 12,
        padding: "10px 12px",
        textAlign: "center",
      }}
    >
      <div
        style={{
          fontSize: 9,
          color: "#6B7280",
          letterSpacing: "0.14em",
          marginBottom: 4,
          textTransform: "uppercase",
        }}
      >
        {label}
      </div>
      <div
        style={{
          fontSize: 20,
          fontWeight: 700,
          fontFamily: "'JetBrains Mono',monospace",
          color,
          lineHeight: 1,
        }}
      >
        {value}
      </div>
      {sub && (
        <div style={{ fontSize: 9, color: "#4B5563", marginTop: 3 }}>{sub}</div>
      )}
    </div>
  );
}

// ── Main ──────────────────────────────────────────────────────────────────────
export default function MemoryEcho() {
  const [screen, setScreen] = useState("home");
  const [lvlIdx, setLvlIdx] = useState(0);
  const [deck, setDeck] = useState([]);
  const [flipped, setFlipped] = useState([]);
  const [matched, setMatched] = useState(0);
  const [moves, setMoves] = useState(0);
  const [timeLeft, setTimeLeft] = useState(90);
  const [phase, setPhase] = useState("idle"); // idle | playing | peek | over | win
  const [peekCD, setPeekCD] = useState(0);
  const [wrongIds, setWrongIds] = useState([]);
  const [shaking, setShaking] = useState(false);
  const [bests, setBests] = useState(() => {
    try {
      return JSON.parse(localStorage.getItem("me2_best") || "{}");
    } catch {
      return {};
    }
  });
  const [cardSize, setCardSize] = useState(64);
  const checking = useRef(false);
  const timerRef = useRef(null);
  const wrapRef = useRef(null);

  const level = LEVELS[lvlIdx];

  // Responsive card size
  useEffect(() => {
    function recalc() {
      const available = Math.min(
        (wrapRef.current?.clientWidth || window.innerWidth) - 32,
        500,
      );
      const gap = (level.cols - 1) * 8;
      const sz = Math.floor((available - gap) / level.cols);
      setCardSize(Math.max(Math.min(sz, 88), 46));
    }
    recalc();
    window.addEventListener("resize", recalc);
    return () => window.removeEventListener("resize", recalc);
  }, [lvlIdx, screen]);

  // Start game
  const startGame = useCallback((idx) => {
    clearInterval(timerRef.current);
    checking.current = false;
    const lvl = LEVELS[idx];
    setDeck(buildDeck(lvl.pairs));
    setFlipped([]);
    setMatched(0);
    setMoves(0);
    setTimeLeft(lvl.time);
    setPhase("playing");
    setPeekCD(0);
    setWrongIds([]);
    setShaking(false);
    setScreen("game");
  }, []);

  // Timer tick
  useEffect(() => {
    if (phase !== "playing") {
      clearInterval(timerRef.current);
      return;
    }
    timerRef.current = setInterval(() => {
      setTimeLeft((t) => {
        if (t <= 1) {
          clearInterval(timerRef.current);
          setPhase("over");
          setScreen("result");
          return 0;
        }
        return t - 1;
      });
    }, 1000);
    return () => clearInterval(timerRef.current);
  }, [phase]);

  // Peek cooldown
  useEffect(() => {
    if (peekCD <= 0) return;
    const id = setInterval(() => setPeekCD((c) => Math.max(0, c - 1)), 1000);
    return () => clearInterval(id);
  }, [peekCD]);

  // Flip card
  const handleFlip = useCallback(
    (id) => {
      if (checking.current || phase !== "playing" || flipped.length >= 2)
        return;
      const newFlipped = [...flipped, id];
      setDeck((d) => d.map((c) => (c.id === id ? { ...c, flipped: true } : c)));
      setFlipped(newFlipped);

      if (newFlipped.length === 2) {
        setMoves((m) => m + 1);
        checking.current = true;
        const cards = newFlipped.map((fid) => deck.find((c) => c.id === fid));

        setTimeout(() => {
          if (cards[0].emoji === cards[1].emoji) {
            setDeck((d) =>
              d.map((c) =>
                newFlipped.includes(c.id)
                  ? { ...c, matched: true, flipped: true }
                  : c,
              ),
            );
            setFlipped([]);
            checking.current = false;
            setMatched((prev) => {
              const nm = prev + 1;
              if (nm === level.pairs) {
                clearInterval(timerRef.current);
                setPhase("win");
                // Save best
                setBests((b) => {
                  const key = level.id;
                  const existing = b[key];
                  const newMoves = moves + 1;
                  if (!existing || newMoves < existing.moves) {
                    const updated = { ...b, [key]: { moves: newMoves } };
                    try {
                      localStorage.setItem("me2_best", JSON.stringify(updated));
                    } catch {}
                    return updated;
                  }
                  return b;
                });
                setTimeout(() => setScreen("result"), 500);
              }
              return nm;
            });
          } else {
            setWrongIds(newFlipped);
            setShaking(true);
            setTimeout(() => {
              setDeck((d) =>
                d.map((c) =>
                  newFlipped.includes(c.id) ? { ...c, flipped: false } : c,
                ),
              );
              setFlipped([]);
              setWrongIds([]);
              setShaking(false);
              checking.current = false;
            }, 750);
          }
        }, 650);
      }
    },
    [flipped, deck, phase, level, moves],
  );

  // Peek
  const handlePeek = () => {
    if (peekCD > 0 || phase !== "playing") return;
    clearInterval(timerRef.current);
    setPhase("peek");
    setDeck((d) => d.map((c) => (c.matched ? c : { ...c, flipped: true })));
    setPeekCD(20);
    setTimeout(() => {
      setDeck((d) =>
        d.map((c) =>
          c.matched ? c : flipped.includes(c.id) ? c : { ...c, flipped: false },
        ),
      );
      setPhase("playing");
    }, 1400);
  };

  const pct = timeLeft / level.time;
  const tColor = pct > 0.5 ? "#8B5CF6" : pct > 0.25 ? "#F59E0B" : "#EF4444";
  const isWin = phase === "win";
  const calcScore = () =>
    Math.max(
      0,
      matched * 100 + timeLeft * 5 - Math.max(0, moves - level.pairs) * 8,
    );

  // ── Render ───────────────────────────────────────────────────────────────────
  return (
    <>
      <style>{GLOBAL_CSS}</style>
      <div className="me-root">
        {/* ── HOME ── */}
        {screen === "home" && (
          <div
            className="screen-anim"
            style={{
              width: "100%",
              maxWidth: 400,
              display: "flex",
              flexDirection: "column",
              gap: 20,
            }}
          >
            {/* Logo block */}
            <div style={{ textAlign: "center", marginBottom: 4 }}>
              <div style={{ fontSize: 40, marginBottom: 12 }}>🧠</div>
              <h1
                style={{
                  fontSize: "2.2rem",
                  fontWeight: 700,
                  color: "#F1EFFE",
                  letterSpacing: "-0.02em",
                  lineHeight: 1.1,
                }}
              >
                Memory Echo
              </h1>
              <p
                style={{
                  fontSize: 13,
                  color: "#6B7280",
                  marginTop: 8,
                  lineHeight: 1.7,
                }}
              >
                Match all pairs before time runs out
              </p>
            </div>

            {/* Level selector */}
            <div
              style={{
                background: "#161923",
                border: "1px solid rgba(255,255,255,0.07)",
                borderRadius: 20,
                overflow: "hidden",
              }}
            >
              <div
                style={{
                  padding: "14px 18px",
                  borderBottom: "1px solid rgba(255,255,255,0.06)",
                }}
              >
                <span
                  style={{
                    fontSize: 10,
                    fontWeight: 600,
                    color: "#6B7280",
                    letterSpacing: "0.16em",
                  }}
                >
                  DIFFICULTY
                </span>
              </div>
              {LEVELS.map((lvl, i) => {
                const sel = lvlIdx === i;
                const b = bests[lvl.id];
                return (
                  <button
                    key={lvl.id}
                    onClick={() => setLvlIdx(i)}
                    style={{
                      width: "100%",
                      display: "flex",
                      alignItems: "center",
                      gap: 14,
                      padding: "14px 18px",
                      background: sel ? "rgba(139,92,246,0.12)" : "transparent",
                      border: "none",
                      borderBottom:
                        i < LEVELS.length - 1
                          ? "1px solid rgba(255,255,255,0.04)"
                          : "none",
                      cursor: "pointer",
                      textAlign: "left",
                      transition: "background 0.2s",
                    }}
                  >
                    <div
                      style={{
                        width: 8,
                        height: 8,
                        borderRadius: "50%",
                        flexShrink: 0,
                        background: sel ? "#8B5CF6" : "#2A2D3A",
                        boxShadow: sel ? "0 0 8px #8B5CF6" : "none",
                        transition: "all 0.2s",
                      }}
                    />
                    <div style={{ flex: 1 }}>
                      <div
                        style={{
                          fontSize: 13,
                          fontWeight: 600,
                          color: sel ? "#C4B5FD" : "#D1D5DB",
                          letterSpacing: "0.02em",
                        }}
                      >
                        {lvl.label}
                      </div>
                      <div
                        style={{ fontSize: 11, color: "#6B7280", marginTop: 1 }}
                      >
                        {lvl.pairs} pairs · {lvl.time}s
                      </div>
                    </div>
                    {b && (
                      <div style={{ textAlign: "right" }}>
                        <div style={{ fontSize: 9, color: "#6B7280" }}>
                          Best
                        </div>
                        <div
                          style={{
                            fontSize: 11,
                            fontFamily: "'JetBrains Mono',monospace",
                            color: "#F59E0B",
                            fontWeight: 600,
                          }}
                        >
                          {b.moves} moves
                        </div>
                      </div>
                    )}
                    {sel && (
                      <div style={{ color: "#8B5CF6", fontSize: 14 }}>✓</div>
                    )}
                  </button>
                );
              })}
            </div>

            {/* How to play */}
            <div
              style={{
                background: "rgba(139,92,246,0.07)",
                border: "1px solid rgba(139,92,246,0.15)",
                borderRadius: 14,
                padding: "14px 18px",
                display: "flex",
                flexDirection: "column",
                gap: 8,
              }}
            >
              {[
                { icon: "👁", text: "Tap cards to flip them over" },
                { icon: "🔮", text: "Find matching emoji pairs" },
                { icon: "⚡", text: "Use Peek to reveal all briefly" },
              ].map(({ icon, text }) => (
                <div
                  key={text}
                  style={{ display: "flex", alignItems: "center", gap: 10 }}
                >
                  <span style={{ fontSize: 16, flexShrink: 0 }}>{icon}</span>
                  <span
                    style={{ fontSize: 12, color: "#9CA3AF", lineHeight: 1.5 }}
                  >
                    {text}
                  </span>
                </div>
              ))}
            </div>

            {/* Play button */}
            <button
              onClick={() => startGame(lvlIdx)}
              style={{
                width: "100%",
                padding: "15px",
                borderRadius: 14,
                border: "none",
                background: "linear-gradient(135deg,#7C3AED,#9D4EDD)",
                color: "#fff",
                fontSize: 14,
                fontWeight: 600,
                fontFamily: "'Sora',sans-serif",
                letterSpacing: "0.06em",
                cursor: "pointer",
                boxShadow: "0 6px 28px rgba(124,58,237,0.4)",
              }}
            >
              Start Playing
            </button>
          </div>
        )}

        {/* ── GAME ── */}
        {screen === "game" && (
          <div
            ref={wrapRef}
            className="screen-anim"
            style={{
              width: "100%",
              maxWidth: 520,
              display: "flex",
              flexDirection: "column",
              alignItems: "center",
              gap: 14,
            }}
          >
            {/* Top nav */}
            <div
              style={{
                width: "100%",
                display: "flex",
                alignItems: "center",
                justifyContent: "space-between",
              }}
            >
              <button
                onClick={() => {
                  clearInterval(timerRef.current);
                  setScreen("home");
                }}
                style={{
                  background: "rgba(255,255,255,0.04)",
                  border: "1px solid rgba(255,255,255,0.07)",
                  borderRadius: 10,
                  padding: "7px 14px",
                  color: "#9CA3AF",
                  fontSize: 12,
                  cursor: "pointer",
                  fontFamily: "'Sora',sans-serif",
                }}
              >
                ← Back
              </button>

              <div
                style={{
                  background: "rgba(139,92,246,0.12)",
                  border: "1px solid rgba(139,92,246,0.25)",
                  borderRadius: 999,
                  padding: "5px 14px",
                  fontSize: 10,
                  fontWeight: 600,
                  color: "#A78BFA",
                  letterSpacing: "0.14em",
                }}
              >
                {level.label.toUpperCase()}
              </div>

              <button
                onClick={() => startGame(lvlIdx)}
                style={{
                  background: "rgba(255,255,255,0.04)",
                  border: "1px solid rgba(255,255,255,0.07)",
                  borderRadius: 10,
                  padding: "7px 14px",
                  color: "#9CA3AF",
                  fontSize: 12,
                  cursor: "pointer",
                  fontFamily: "'Sora',sans-serif",
                }}
              >
                ↺ Restart
              </button>
            </div>

            {/* Stats row */}
            <div
              style={{
                width: "100%",
                display: "flex",
                gap: 8,
                alignItems: "center",
              }}
            >
              <StatBox
                label="Matched"
                value={`${matched}/${level.pairs}`}
                color="#34D399"
              />
              <TimerArc left={timeLeft} total={level.time} />
              <StatBox label="Moves" value={moves} color="#C4B5FD" />
            </div>

            {/* Progress bar */}
            <div
              style={{
                width: "100%",
                height: 3,
                background: "#1E2130",
                borderRadius: 2,
                overflow: "hidden",
              }}
            >
              <div
                style={{
                  height: "100%",
                  borderRadius: 2,
                  width: `${(matched / level.pairs) * 100}%`,
                  background: "linear-gradient(90deg,#7C3AED,#C084FC)",
                  transition: "width 0.4s cubic-bezier(0.16,1,0.3,1)",
                }}
              />
            </div>

            {/* Pair dots */}
            <div
              style={{
                display: "flex",
                gap: 5,
                flexWrap: "wrap",
                justifyContent: "center",
              }}
            >
              {Array.from({ length: level.pairs }).map((_, i) => (
                <div
                  key={i}
                  style={{
                    width: 9,
                    height: 9,
                    borderRadius: "50%",
                    background: i < matched ? "#34D399" : "#1E2130",
                    border: `1.5px solid ${i < matched ? "#34D399" : "#2A2D3A"}`,
                    transition: "all 0.3s",
                    boxShadow:
                      i < matched ? "0 0 5px rgba(52,211,153,0.5)" : "none",
                  }}
                />
              ))}
            </div>

            {/* Card grid */}
            <div
              style={{
                display: "grid",
                gridTemplateColumns: `repeat(${level.cols}, ${cardSize}px)`,
                gap: 8,
                animation: shaking ? "wrongShake 0.4s ease" : "none",
              }}
            >
              {deck.map((card) => (
                <Card
                  key={card.id}
                  card={card}
                  onClick={handleFlip}
                  disabled={phase !== "playing" || checking.current}
                  size={cardSize}
                  isWrong={wrongIds.includes(card.id)}
                />
              ))}
            </div>

            {/* Peek */}
            <button
              onClick={handlePeek}
              disabled={peekCD > 0 || phase !== "playing"}
              style={{
                padding: "11px 28px",
                borderRadius: 12,
                border: `1.5px solid ${peekCD > 0 ? "rgba(255,255,255,0.05)" : "rgba(139,92,246,0.4)"}`,
                background:
                  peekCD > 0
                    ? "rgba(255,255,255,0.02)"
                    : "rgba(139,92,246,0.1)",
                color: peekCD > 0 ? "#374151" : "#A78BFA",
                fontSize: 12,
                fontWeight: 600,
                fontFamily: "'Sora',sans-serif",
                letterSpacing: "0.06em",
                cursor: peekCD > 0 ? "not-allowed" : "pointer",
                display: "flex",
                alignItems: "center",
                gap: 8,
                transition: "all 0.2s",
              }}
            >
              <span style={{ fontSize: 16 }}>👁</span>
              {peekCD > 0 ? `Peek — ${peekCD}s cooldown` : "Peek All Cards"}
            </button>

            {/* Status hint */}
            <p
              style={{
                fontSize: 11,
                color: "#4B5563",
                letterSpacing: "0.08em",
                textAlign: "center",
              }}
            >
              {phase === "peek"
                ? "Memorize quickly!"
                : phase === "playing"
                  ? `${level.pairs - matched} pairs remaining · tap any hidden card`
                  : ""}
            </p>
          </div>
        )}

        {/* ── RESULT ── */}
        {screen === "result" && (
          <div
            className="screen-anim"
            style={{
              width: "100%",
              maxWidth: 380,
              display: "flex",
              flexDirection: "column",
              gap: 22,
              alignItems: "center",
            }}
          >
            {/* Icon + title */}
            <div style={{ textAlign: "center" }}>
              <div style={{ fontSize: 52, lineHeight: 1, marginBottom: 14 }}>
                {isWin ? "🎉" : "⏳"}
              </div>
              <div
                style={{
                  display: "inline-block",
                  padding: "4px 14px",
                  borderRadius: 999,
                  background: isWin
                    ? "rgba(52,211,153,0.12)"
                    : "rgba(239,68,68,0.12)",
                  border: `1px solid ${isWin ? "rgba(52,211,153,0.3)" : "rgba(239,68,68,0.3)"}`,
                  fontSize: 10,
                  fontWeight: 600,
                  color: isWin ? "#34D399" : "#EF4444",
                  letterSpacing: "0.18em",
                  marginBottom: 12,
                }}
              >
                {isWin ? "COMPLETE" : "TIME'S UP"}
              </div>
              <h2
                style={{
                  fontSize: "1.9rem",
                  fontWeight: 700,
                  color: "#F1EFFE",
                  letterSpacing: "-0.02em",
                }}
              >
                {isWin ? "Well done!" : "Nice try!"}
              </h2>
              <p style={{ fontSize: 13, color: "#6B7280", marginTop: 6 }}>
                {isWin
                  ? `Matched all ${level.pairs} pairs`
                  : `Found ${matched} of ${level.pairs} pairs`}
              </p>
            </div>

            {/* Stats card */}
            <div
              style={{
                width: "100%",
                background: "#161923",
                border: "1px solid rgba(255,255,255,0.07)",
                borderRadius: 20,
                overflow: "hidden",
              }}
            >
              {[
                { label: "Score", value: calcScore(), color: "#C084FC" },
                { label: "Level", value: level.label, color: "#E8E6F0" },
                {
                  label: "Pairs Found",
                  value: `${matched}/${level.pairs}`,
                  color: "#34D399",
                },
                { label: "Total Moves", value: moves, color: "#E8E6F0" },
                { label: "Time Left", value: fmt(timeLeft), color: tColor },
                ...(bests[level.id]
                  ? [
                      {
                        label: "Personal Best",
                        value: `${bests[level.id].moves} moves`,
                        color: "#F59E0B",
                      },
                    ]
                  : []),
              ].map(({ label, value, color }, i, arr) => (
                <div
                  key={label}
                  style={{
                    display: "flex",
                    justifyContent: "space-between",
                    alignItems: "center",
                    padding: "13px 20px",
                    borderBottom:
                      i < arr.length - 1
                        ? "1px solid rgba(255,255,255,0.05)"
                        : "none",
                  }}
                >
                  <span
                    style={{
                      fontSize: 12,
                      color: "#6B7280",
                      letterSpacing: "0.06em",
                    }}
                  >
                    {label}
                  </span>
                  <span
                    style={{
                      fontSize: 13,
                      fontWeight: 600,
                      fontFamily: "'JetBrains Mono',monospace",
                      color,
                    }}
                  >
                    {value}
                  </span>
                </div>
              ))}
            </div>

            {/* New best */}
            {isWin && bests[level.id]?.moves === moves && (
              <div
                style={{
                  display: "flex",
                  alignItems: "center",
                  gap: 8,
                  background: "rgba(245,158,11,0.1)",
                  border: "1px solid rgba(245,158,11,0.25)",
                  borderRadius: 999,
                  padding: "7px 18px",
                }}
              >
                <span>🏆</span>
                <span
                  style={{
                    fontSize: 11,
                    fontWeight: 600,
                    color: "#F59E0B",
                    letterSpacing: "0.1em",
                  }}
                >
                  NEW PERSONAL BEST!
                </span>
              </div>
            )}

            {/* Buttons */}
            <div
              style={{
                width: "100%",
                display: "flex",
                flexDirection: "column",
                gap: 10,
              }}
            >
              <button
                onClick={() => startGame(lvlIdx)}
                style={{
                  width: "100%",
                  padding: "14px",
                  borderRadius: 13,
                  border: "none",
                  background: "linear-gradient(135deg,#7C3AED,#9D4EDD)",
                  color: "#fff",
                  fontSize: 13,
                  fontWeight: 600,
                  fontFamily: "'Sora',sans-serif",
                  letterSpacing: "0.06em",
                  cursor: "pointer",
                  boxShadow: "0 6px 24px rgba(124,58,237,0.35)",
                }}
              >
                Play Again
              </button>
              <div style={{ display: "flex", gap: 10 }}>
                <button
                  onClick={() => setScreen("home")}
                  style={{
                    flex: 1,
                    padding: "12px",
                    borderRadius: 12,
                    border: "1.5px solid rgba(255,255,255,0.08)",
                    background: "rgba(255,255,255,0.03)",
                    color: "#9CA3AF",
                    fontSize: 12,
                    fontFamily: "'Sora',sans-serif",
                    letterSpacing: "0.06em",
                    cursor: "pointer",
                  }}
                >
                  Home
                </button>
                {isWin && lvlIdx < LEVELS.length - 1 && (
                  <button
                    onClick={() => {
                      const n = lvlIdx + 1;
                      setLvlIdx(n);
                      startGame(n);
                    }}
                    style={{
                      flex: 1,
                      padding: "12px",
                      borderRadius: 12,
                      border: "1.5px solid rgba(139,92,246,0.35)",
                      background: "rgba(139,92,246,0.1)",
                      color: "#A78BFA",
                      fontSize: 12,
                      fontFamily: "'Sora',sans-serif",
                      letterSpacing: "0.06em",
                      cursor: "pointer",
                    }}
                  >
                    Next Level →
                  </button>
                )}
              </div>
            </div>
          </div>
        )}
      </div>
    </>
  );
}
